/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include "typedefs.h"                    // Required
#include "usb.h"                         // Required
#include "io_cfg.h"                      // Required
#include <math.h>

#define VER_HIGH	1
#define VER_LOW		1

//#include "usb_compile_time_validation.h" // Optional

/** C O N F I G U R A T I O N ************************************************/

#if   defined(__18F4550)||defined(__18F4455)|| \
      defined(__18F2550)||defined(__18F2455)|| \
      defined(__18F4553)||defined(__18F4458)|| \
      defined(__18F2553)||defined(__18F2458)

// Note: Some of the below configuration bits are commented out
// to prevent build errors with some of the above listed devices.
// For example, on the PIC18F4458 CP3, WRT3, and EBTR3 don't exist.

#pragma config PLLDIV   = 5       		// (20 MHz input)
#pragma config CPUDIV   = OSC1_PLL2		//
#pragma config USBDIV   = 2       		// Clock source from 96MHz PLL/2
#pragma config FOSC     = HSPLL_HS		//
#pragma config FCMEN    = OFF			//
#pragma config IESO     = OFF
#pragma config PWRT     = OFF
#pragma config BOR      = ON
#pragma config BORV     = 3
#pragma config VREGEN   = ON
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = ON
#pragma config LPT1OSC  = ON
#pragma config PBADEN   = OFF
#pragma config CCP2MX   = OFF
#pragma config STVREN   = ON
#pragma config LVP      = ON
//#pragma config ICPRT    = OFF       	// Dedicated In-Circuit Debug/Programming
#pragma config XINST    = OFF       	// Extended Instruction Set
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CP2      = OFF
//#pragma config CP3      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRT2     = OFF
//#pragma config WRT3     = OFF
#pragma config WRTB     = ON      		// Boot Block Write Protection ON
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTR2    = OFF
//#pragma config EBTR3    = OFF
#pragma config EBTRB    = OFF

#endif

/** V A R I A B L E S ********************************************************/
#pragma udata

/** P R I V A T E  P R O T O T Y P E S ***************************************/
//
#define T2_PERIOD_DEFAULT 			82
#define MAX_PWM_DUTY_LOW_POWER		20
#define MIN_PWM_DUTY				5
#define EXTRA_LOW_POWER_LV_TRIP		2.17			
//
/** V E C T O R  R E M A P P I N G *******************************************/
void myISR(void);
void myISRHigh(void);
void disAdec(int, int);
void disA(int, int);
void disWdec(int);
unsigned int readAN(unsigned char);
void setPWMDuty(int);
int WriteEEPROM(int, int);
int ReadEEPROM(int);

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}

#pragma code
/** D E C L A R A T I O N S **************************************************/
#pragma code

void
DelayMs(int delay)
{
	unsigned int n;

	while(delay--) {
		for(n = 0; n < 500; n++) {
		// the following must be around a 2us delay...	
			_asm
				nop
				nop
				nop
				nop
			_endasm
		}
	}
}

rom char sevenSegment[16]={ 0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7D, 0x07, 0x7F, 0x6F, 0x77, 0x7C, 0x39, 0x5E, 0x79, 0x71 };

//#define NUMBER_OF_CENTURIES 	7
#define NUMBER_OF_MONTHS		12
#define YEAR_OFFSET				1900
#define MIN_YEAR				(1900-YEAR_OFFSET)
#define MAX_YEAR				(2599-YEAR_OFFSET)
#define MODE_MODULUS			6

//rom int CenturiesTable[NUMBER_OF_CENTURIES]=
//						{ 	0, 		// 1900-1999
//							6,		// 2000-2099
//							4,		// 2100-2199
//							2,		// 2200-2299
//							0,		// 2300-2399
//							6,		// 2400-2499
//							4		// 2500-2599
//						  };
//
//rom int MonthsTable[NUMBER_OF_MONTHS]=
//						{ 	0,			// Jan
//							3,			// Feb
//							3,			// Mar
//							6,			// Apr
//							1,			// May
//							4,			// Jun
//							6,			// Jul
//							2,			// Aug
//							5,			// Sep
//							0,			// Oct
//							3,			// Nov
//							5			// Dec
//						};

rom int DaysOfTheMonth[NUMBER_OF_MONTHS]={ 	
	31,			// Jan
	28,			// Feb
	31,			// Mar
	30,			// Apr
	31,			// May
	30,			// Jun
	31,			// Jul
	31,			// Aug
	30,			// Sep
	31,			// Oct
	30,			// Nov
	31			// Dec
};

char decimalPoint;
char digits[4];
int screenCounter, screenPeriod, screenUpdateOn;
unsigned long myCounter;
unsigned int hlvdset;
float setVoltageLevelHLVD;
int usbRegAssumedVoltage;
// **** Actual USB Clock Global Variables Communicated ******
int senseresistor;
int an0, an1;
int RTCcountdown;
int RTCseconds;
int displayCount;
int RTCminutes;
int RTChours;
int RTCwday;
int RTCmday;
int RTCyday;
int RTCisdst;
int RTCmonth;
int RTCyear;
int verhigh;
int verlow;
int lowPowerPWMDuty;
int timeOutPeriod;
int lvtripPoint;			// in mV
int lastseconds;
int lastminutes;
int lasthours;
int lastwday;
int lastyday;
int lastmday;
int lastisdst;
int lastmonth;
int lastyear;
int sinceseconds;
int sinceminutes;
int sincehours;
int sinceday;
int sincemonth;
int sinceyear;
int clockmode;
int requestReset;
int powermode;
//
int screenRefreshes;
int currentPWMDuty;
int lowPowerModeEnable;
//
void goToLowPower(void)
{
	if((lowPowerModeEnable!=0)&&(currentPWMDuty==0))
	{
		powermode|=1;
		OSCCONbits.IDLEN=1;
		OSCCONbits.SCS0=1;
		OSCCONbits.SCS1=1;
		Sleep();
	} 
	else powermode&=~1;
}
//
void refreshScreen(void)
{
	int i, j;
	int a0, b0, c0, d0, e0;				// for the LCD pins
	int a1, b1, c1, d1, e1;				// for the latches

	screenRefreshes++;
	screenCounter++;
	// build up a0 and a1 outputs
	a0=0;
	a1=0;
	if((digits[0] & 0x02)!=0)a0|=0x04;			// RA2 = 1B
	if((digits[0] & 0x01)!=0)a0|=0x10;			// RA4 = 1A
	if((digits[0] & 0x20)!=0)a0|=0x20; 			// RA5 = 1F
	//
	if((decimalPoint & 0x08)!=0)a1|=0x04;		// latched RA2 = DP1
	if((digits[0] & 0x04)!=0)a1|=0x10; 			// latched RA4 = 1C
	if((digits[0] & 0x08)!=0)a1|=0x20; 			// latched RA5 = 1D
	//

	// build up b0 and b1 outputs
	b0=0;
	b1=0;
	if((decimalPoint & 0x01)!=0)b0|=0x01;		// RB0 = Col
	if((digits[1]& 0x02)!=0)b0|=0x02;			// RB1 = 2B
	if((digits[1]& 0x01)!=0)b0|=0x04;			// RB2 = 2A
	if((digits[1]& 0x20)!=0)b0|=0x08;			// RB3 = 2F
	if((digits[1]& 0x40)!=0)b0|=0x10;			// RB4 = 2G
	//
	if((digits[2] & 0x10)!=0)b1|=0x01;			// latched RB0 = 3E
	if((decimalPoint & 0x04)!=0)b1|=0x02;		// latched RB1 = DP2
	if((digits[1]& 0x04)!=0)b1|=0x04;			// latched RB2 = 2C
	if((digits[1]& 0x08)!=0)b1|=0x08;			// latched RB3 = 2D
	if((digits[1]& 0x10)!=0)b1|=0x10;			// latched RB4 = 2E
	//

	// build up PORTC output
	c0=0;
	c1=0;
	if((digits[3] & 0x20)!=0)c0|=0x40;			// RC6 = 4F
	if((digits[3] & 0x40)!=0)c0|=0x80;			// RC7 = 4G
	//
	if((digits[3] & 0x04)!=0)c1|=0x40;			// latched RC6 = 4C
	if((digits[3] & 0x08)!=0)c1|=0x80;			// latched RC7 = 4D
	//

	// build up PORTD output
	d0=0;
	d1=0;
	if((digits[3] & 0x01)!=0)d0|=0x04;			// RD2 = 4A
	if((digits[2] & 0x02)!=0)d0|=0x10;			// RD4 = 3B
	if((digits[2] & 0x01)!=0)d0|=0x20;			// RD5 = 3A
	if((digits[2] & 0x20)!=0)d0|=0x40;			// RD6 = 3F
	if((digits[2] & 0x40)!=0)d0|=0x80;			// RD7 = 3G
	//
	if((digits[3] & 0x02)!=0)d1|=0x04;			// latched RD2 = 4B
	if((digits[3] & 0x10)!=0)d1|=0x10;			// latched RD4 = 4E
	if((decimalPoint & 0x02)!=0)d1|=0x20;		// latched RD5 = DP3
	if((digits[2] & 0x04)!=0)d1|=0x40;			// latched RD6 = 3C
	if((digits[2] & 0x08)!=0)d1|=0x80;			// latched RD7 = 3D

	// build up PORTE output
	e0=0;
	e1=0;
	if((digits[0] & 0x40)!=0)e0|=0x01;			// RE0 = 1G
	//
	if((digits[0] & 0x10)!=0)e1|=0x01;			// latched RE0 = 1E
	//
	if((screenCounter & 0x01)!=0)
	{ 
				i=0xFF;
	} 
	else 
	{
				i=0x00;
	}
	a0= a0 ^ i;
	a1= a1 ^ i;
	b0= b0 ^ i;
	b1= b1 ^ i;
	c0= c0 ^ i;
	c1= c1 ^ i;
	d0= d0 ^ i;
	d1= d1 ^ i;
	e0= e0 ^ i;
	e1= e1 ^ i;
	
	if(PORTCbits.RC2==1){ c0|=0x04; c1|=0x04; } else { c0&=~0x04; c1&=~0x04; }
	if(PORTBbits.RB6==1){ b0|=0x40; b1|=0x40; } else { b0&=~0x40; b1&=~0x40; }

	e1 = e1 | 0x02;					// RE1 = LE of IC3
	d1 = d1 | 0x08;					// RD3 = LE of IC2
	e0 = e0 & (~0x02);				// RE1 = LE of IC3
	d0 = d0 & (~0x08);				// RD3 = LE of IC2

	PORTA=a1;
	PORTB=b1;
	PORTC=c1;
	PORTD=d1;
	PORTE=e1;
		_asm
			nop
			nop
			nop
			nop
		_endasm
	PORTEbits.RE1=0;
	PORTDbits.RD3=0;
		_asm
			nop
			nop
			nop
			nop
		_endasm
	PORTE=e0;
	PORTD=d0;
	PORTC=c0;
	PORTB=b0;
	PORTA=a0;
}


void doHLVDInterrupt(void)
{
	hlvdset++;
	if(currentPWMDuty!=0)setPWMDuty(currentPWMDuty/2);
	else
	{
		// enter extra low power conservation power mode!
		powermode|=0x02;			// entering extra low power conservation mode!
		
	}
}

int timeOut(void)
{
	if(RTCcountdown>0)return FALSE; else return TRUE;
}

void setTimeOut(int seconds)
{
	RTCcountdown=seconds;
}

int isLeapYear(int iyear)
{
	// returns TRUE if iyear is a leap year, FALSE otherwise
	int i, j;
	i=(iyear%4);
	if(i!=0)return 0;	// since not divisible by 4
	i=(iyear%100);
	if(i!=0)return 1;
	j=(iyear%400);
	if(j==0)return 1; else return 0;
}


void RTCUpdater(void)
{
	int i, j;

	RTCseconds++;
	if(RTCcountdown>0)RTCcountdown--; else RTCcountdown=0;
	if(RTCseconds>59)
	{ 	
	RTCseconds=0; 
	RTCminutes++; 
	}
	if(RTCminutes>59)
	{
	RTChours++;
	RTCminutes=0;
	}
	if(RTChours>23)
	{
	RTChours=0;
	RTCwday=((RTCwday +1)%7);
	RTCmday++;
	RTCyday++;
	i=isLeapYear(YEAR_OFFSET+RTCyear);
	j=DaysOfTheMonth[RTCmonth];
	if((RTCmonth==1)&&(i==1))j++;				// a leap year for Feb
	if(RTCmday>j)
				{
				RTCmday=1;
				RTCmonth++;	
				if(RTCmonth>=NUMBER_OF_MONTHS)
				{
				RTCmonth=0;
				RTCyear++;
				RTCyday=0;
				}
		}	
	}
}

void ScreenUpdater(void)
{
		int i;
		if(clockmode==0)
		{
			disAdec(RTCminutes,0);
			disAdec(RTCseconds,2);
			if((RTCseconds & 0x01)!=0)decimalPoint=0; else decimalPoint=1;
		} else
		if(clockmode==1)
\		{
			disAdec(RTChours, 0);
			disAdec(RTCminutes,2);
			if((RTCseconds & 0x02)!=0)decimalPoint=0; else decimalPoint=1;
		} else
		if(clockmode==2)
		{
			// display date mode
			i=displayCount % 4;
			if(i==0)
			{
			digits[0]=0x5E;		// d
			digits[1]=0x77;		// A
			digits[2]=0x6E;		// Y			
			digits[3]=0;
			decimalPoint=0;
			} else
			{		
			disAdec(RTCmday,0);
			disAdec(RTCmonth+1,2);
			decimalPoint=4;
			}
			displayCount++;

		} else
		if(clockmode==3)
		{
			// display the year
			i=displayCount % 4;
			if(i==0)
			{
			digits[0]=0x6E;		// Y
			digits[1]=0x79;		// E
			digits[2]=0x77;		// A
			digits[3]=0x50;		// R
			decimalPoint=0;
			} else
			{
			disWdec(RTCyear+YEAR_OFFSET);
			decimalPoint=0;
			}
			displayCount++;
		} else
		if(clockmode==4)
		{
			// display the voltage
			disWdec(timeOutPeriod);
			decimalPoint=0;
		} else
		if(clockmode==5)
		{
			// display the charge rate
			disWdec(an1);
			decimalPoint=0;
		} else
		if(clockmode==6)
		{
			digits[0]=0x40;
			digits[1]=0x40;
			digits[2]=0x40;
			digits[3]=0x40;
			decimalPoint=0;
		} else
		{
			disAdec(RTCminutes,0);
			disAdec(RTCseconds,2);
		}
		an0=readAN(0);
		an1=readAN(1); 
		if(PORTBbits.RB6!=0)
		{
				// button not pressed					
		} else 
		{ 
				// button pressed
				setTimeOut(timeOutPeriod);
				setPWMDuty(lowPowerPWMDuty);
		}
	
}

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	OSCCONbits.SCS1=0;
	OSCCONbits.SCS0=0;
	TMR1H+=0x80;				
	RTCUpdater();
	if(screenUpdateOn!=0)
	{
	ScreenUpdater();
	}					
	PIR1bits.TMR1IF=0;	
	goToLowPower();
}

#pragma interruptlow myISR
void myISR(void)
{
	OSCCONbits.SCS0=0;
	OSCCONbits.SCS1=0;
	if(PIR2bits.TMR3IF==1)
		{ 	
			refreshScreen();
			TMR3H=screenPeriod>>8;
			TMR3L=screenPeriod;
			if(!timeOut()){
			
			} else if(currentPWMDuty>0)setPWMDuty(0.99*currentPWMDuty);
			PIR2bits.TMR3IF=0;
		}
	else
	if(PIR2bits.HLVDIF==1)
		{
			PIR2bits.HLVDIF=0;
			doHLVDInterrupt();
			PIE2bits.HLVDIE=1;			// disable the interrupt!
		}
	USBDriverService();
	BootService();
	if(requestReset!=0){
			_asm
				reset
			_endasm
	}
	goToLowPower();
}

void disAdec(int x, int cursor)
{
	int l;
	l=(x%10);
	digits[cursor]=sevenSegment[0x0F & (((x-l)/10)%10)];
	digits[cursor+1]=sevenSegment[0x0F & l];
}

void disA(int x, int cursor)
{
	// in Hex
	digits[cursor]=sevenSegment[0x0F & (x>>4)];
	digits[cursor+1]=sevenSegment[0x0F & x];
}

void disWdec(int x)
{
	int l;
	l=(x % 10);
	digits[3]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[2]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[1]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[0]=sevenSegment[0x0F & l];
}

rom float hlv_bound[16]={ 2.17, 2.23, 2.36, 2.44, 2.60, 2.79, 2.89, 3.12, 3.39, 3.55, 3.71, 3.90, 4.11, 4.33, 4.59, 10.0 };

float initHLVD(unsigned char mode, float voltage_level)
{
	// mode=0 falling level (use bounds_min levels)
	// mode=1 rising level (use the bounds_max levels)
	// voltage_level = to set
	// returns the actual level set...
	
	int i, k;
	float gg;
	hlvdset=0;
	k=0;					// for default!
	HLVDCONbits.HLVDEN=0;	// disable module
	for(i=0; i<16; i++)
	{
	if( (voltage_level > hlv_bound[i]) && (voltage_level < hlv_bound[i+1]))k=i;
	}
	gg=hlv_bound[k];
	if(mode==0)HLVDCONbits.VDIRMAG=0; else HLVDCONbits.VDIRMAG=1;
	HLVDCON&=0xF0;
	HLVDCON|=(0x0F & k);
	HLVDCONbits.HLVDEN=1;
	PIR2bits.HLVDIF=0;
	IPR2bits.HLVDIP=0;			// low interrupt priority!
	PIE2bits.HLVDIE=1;			//  enable interrupt!
	return gg;
}

void initLCD(void)
{
		int i;
		PORTA=0xFF;
		PORTB=0xFF;
		PORTC=0xFB;
		PORTD=0xFF;
		PORTE=0xFF;
		TRISA&=~0b00110100;
		TRISB&=~0b00011111;
		TRISC&=~0b11000100;			
		TRISD&=~0b11111100;
		TRISE&=~0b00000111;
		for(i=0; i<4; i++)digits[i]=0x40;
		decimalPoint=0;
		//T0CON=0x80;
		//INTCONbits.TMR0IF=0;
		//INTCON2bits.TMR0IP=0;		// low priority for TMR0 interrupt!
		//INTCONbits.TMR0IE=1;
		screenPeriod=0xFD00;
		screenCounter=0;
		screenRefreshes=0;
		T3CON=0x83;
		PIR2bits.TMR3IF=0;
		IPR2bits.TMR3IP=0;
		PIE2bits.TMR3IE=1;
}

int keyPressed(void)
{
	int i;
	i=PORTB;
	i=i & 0xC0;
	i=i >> 6;
	return i;
}

int waitForKey(void)
{
	int i;
	i=0xFF;
	while(i!=0)i=keyPressed();
	while(i==0)i=keyPressed();
	return i;
}

void initRTC(void)
{
		// real time clock using external 32768 Hz crystal.
	TMR1H=0x00;
	TMR1L=0x00;					// to write we write L last to read read L first (16 bit)
	T1CON=0x4F;					
	PIR1bits.TMR1IF=0;
	IPR1bits.TMR1IP=1;			// high interrupt
	DelayMs(100);				// to allow oscillator to stabilize
	PIE1bits.TMR1IE=1;			// enable the interrupt!
}

int initADC(void)
{
	ADCON1=0x0D;			// enable AN0 and AN1 as analog inputs!
	ADCON2=0xA6;
	ADCON0=0x01;
}


void setPWMDuty(int percentage)
{
	// f is a percentage from 0 to 100 %
	int i;
	if(percentage<MIN_PWM_DUTY)percentage=0; else if(percentage>99)percentage=100;
	currentPWMDuty=percentage;
	i=((T2_PERIOD_DEFAULT+1)*percentage)/100;
	if(i<0)i=0; else if(i>0xFF)i=0xFF;
	CCPR1L=i;				// only 8 bits are used...
}

int initPWM(void)
{
	CCP1CON=0;
	T2CON=0;
	PR2=T2_PERIOD_DEFAULT;	// this gives a PWM frequency arounf 30kHz at 20MHz system clock
	setPWMDuty(0);
	TRISC&=~0x04;			// make RC2 an output which is multiplexed with CCP1 (pin 36 of TQFP package)
	T2CON=0x05;				// turn on Timer 2 with 1:2 pre and 1:1 post scaling
	PIR1bits.CCP1IF=0;		// clear interrupt pending
	PIE1bits.CCP1IE=0;		// disable interrupt
	IPR1bits.CCP1IP=0;		// low interrupt priority, if interrupt is enabled
	CCP1CON=0x0C;			// enable PWM mode for CCP1 module
}

void initSystem(int c, float vv, float usbregvoltage)
{
	lowPowerModeEnable=0;
	setVoltageLevelHLVD=initHLVD(c, vv);
	lvtripPoint=(int)(setVoltageLevelHLVD*1000.0);
	INTCON2bits.RBPU=0;
	usbRegAssumedVoltage=(int)(1000.0*usbregvoltage);
	RCONbits.IPEN=1;		// enable prioritized interrupts!
	setTimeOut(0);
	initADC();
	initLCD();
	initRTC();
	initPWM();
	INTCONbits.GIEL=1;		// enable low interrupts!
	INTCONbits.GIE=1;		// enable all interrupts!!
}

unsigned int readAN(unsigned char channel)
{
	unsigned int i;
	ADCON0&=0xC3;
	i=0x3C & ((channel & 0x0F)<<2);
	ADCON0|=i;				// set the channel
	ADCON0|=0x02;
	while((ADCON0 & 0x02)!=0)i++;
	i=ADRESH*256 + ADRESL;
	return i;
}

float readVCC()
{
	// assumes that the output of the 3.3V internal regulator for USB
	// is connected via a 33k ohm resistor to the AN0 input
	float t;
	int i;
	i=readAN(0);
	if(i>0)
	{ 
		t=usbRegAssumedVoltage*1.023000/((float)i);	
	} 
	else
	{
		t=0.0;
	}		
	return t;
}

void wait(void)
{
	_asm
		nop
	_endasm
}

void USBConnect(void)
{
	screenUpdateOn=0;
	digits[0]=0x3E;
	digits[1]=0x6D;
	digits[2]=0x7C;
	setTimeOut(4);			// 3 seconds allowed to connect before timeout
	while(!timeOut())
	{
		digits[3]=sevenSegment[usb_device_state & 0x0F];
		USBDriverService();
		BootService();
	}
}

void setClockMode(int imode)
{
	clockmode=(imode % MODE_MODULUS);
	displayCount=0;
}

void setTimeOutPeriod(int per)
{
	if(per<0)per=5;
	timeOutPeriod=per;
}

void setLowPowerPWMDuty(int y)
{
	if(y<0)y=MAX_PWM_DUTY_LOW_POWER;
	else lowPowerPWMDuty=y;
}

int initSequence(void)
{
	int i;
	
	// these are the defaults
	powermode=0;
	requestReset=0;
	an0=800;
	an1=0;
	senseresistor=1350;			// in mOhms
	screenUpdateOn=0;
	RTCseconds=0;
	RTCminutes=0;
	RTChours=12;
	RTCwday=2;
	RTCmday=1;
	RTCyday=0;
	RTCisdst=0;
	RTCmonth=0;
	RTCyear=2008-YEAR_OFFSET;
	lastseconds=RTCseconds;
	lastminutes=RTCminutes;
	lasthours=RTChours;
	lastwday=RTCwday;
	lastmday=RTCmday;
	lastyday=RTCyday;
	lastisdst=RTCisdst;
	lastyear=RTCyear;
	lastmonth=RTCmonth;
	verhigh=VER_HIGH;
	verlow=VER_LOW;
	setLowPowerPWMDuty(MAX_PWM_DUTY_LOW_POWER);
	setTimeOutPeriod(5);
	setClockMode(0);
	//
 	mInitializeUSBDriver();     // See usbdrv.h
	initSystem(0, 2.5, 3.3);
	USBCheckBusStatus();        // Modified to always enable USB module
	USBConnect();
	screenUpdateOn=1;
	lowPowerModeEnable=1;		// enable Power Saving!
}

void saveAllSettings(void)
{
	// save all settings to EEPROM
}

void restoreAllSettings(void)
{
	// restore all settings from EEPROM

}

void main(void)
{
	initSequence();
	while(1)
	{
		// do nothing...
	}
}

#pragma code user = RM_RESET_VECTOR

/** EOF main.c ***************************************************************/
